#ifndef ACTION_NODES_H
#define ACTION_NODES_H

#include "behaviortree_cpp_v3/bt_factory.h"
#include <ncurses.h> // Used by Supervisor
#include "sequencer/vpKeyboard.hpp"

using namespace BT;

class MyAsyncAction: public CoroActionNode
{
  public:
    MyAsyncAction(const std::string& name):
        CoroActionNode(name, {})
    {}

  private:
    NodeStatus tick() override

    {
        std::cout << name() <<": Started. Send Request to server." << std::endl;

        auto Now = [](){ return std::chrono::high_resolution_clock::now(); };

        TimePoint initial_time = Now();
        TimePoint time_before_reply = initial_time + std::chrono::seconds(5);
        TimePoint yieldCoutTimer = initial_time + std::chrono::seconds(1);

        int count = 0;
        int cptSec = 0;
        bool reply_received = false;

        while( !reply_received )
        {
            if( count++ == 0)
            {
                // call this only once
                std::cout << name() <<": Waiting Reply..." << std::endl;
            }
            // pretend that we received a reply
            if( Now() >= time_before_reply )
            {
                reply_received = true;
            }

            if( !reply_received )
            {
                // set status to RUNNING and "pause/sleep"
                // If halt() is called, we will not resume execution (stack destroyed)
                if(Now() >= yieldCoutTimer){
                    cptSec++;
                    std::cout << cptSec << "s" << std::endl;
                    yieldCoutTimer = Now() + std::chrono::seconds(1);
                }
                setStatusRunningAndYield();
            }
        }

        // This part of the code is never reached if halt() is invoked,
        // only if reply_received == true;
        std::cout << name() <<": Done. 'Waiting Reply' loop repeated "
                  << count << " times" << std::endl;
        cleanup(false);
        return NodeStatus::SUCCESS;
    }

    // you might want to cleanup differently if it was halted or successful
    void cleanup(bool halted)
    {
        if( halted )
        {
            std::cout << name() <<": cleaning up after an halt()\n" << std::endl;
        }
        else{
            std::cout << name() <<": cleaning up after SUCCESS\n" << std::endl;
        }
    }
    void halt() override
    {
        std::cout << name() <<": Halted." << std::endl;
        cleanup(true);
        // Do not forget to call this at the end.
        CoroActionNode::halt();
    }
};

class Supervisor: public CoroActionNode
{
  public:
    void usage(){
        std::cout << name() <<": Started." << std::endl;
        std::cout << "Options:" << std::endl;
        std::cout << "[h] Display this message" << std::endl;
        std::cout << "[s] Start the sequence" << std::endl;
        std::cout << "[p] Pause/resume the sequence" << std::endl;
        std::cout << "[q] Quit the sequence" << std::endl << std::endl;
    }

    Supervisor(const std::string& name):
        CoroActionNode(name, {})
    {
        usage();
        sequenceRunning=false;
    }

  private:
    bool sequenceRunning;
    NodeStatus tick() override
    {   
        int input;
        vpKeyboard keyboard;

        if (keyboard.kbhit()) {
            input = keyboard.getchar();
            // printf("You hit key: %d '%c'\n", input, input);
        }

        switch (input)
        {
        case 'h':
            usage();
            break;
        case 's':
            sequenceRunning=true;
            break;
        case 'p':
            sequenceRunning=!sequenceRunning;
            break;
        case 'q':
            sequenceRunning=false;
            return NodeStatus::FAILURE;
            break;        
        default:
            break;
        }

        if(sequenceRunning)
        {
            return NodeStatus::SUCCESS;
        }

        return NodeStatus::RUNNING;        
    }

    // you might want to cleanup differently if it was halted or successful
    void cleanup(bool halted)
    {
        if( halted )
        {
            std::cout << name() <<": cleaning up after an halt()\n" << std::endl;
        }
        else{
            std::cout << name() <<": cleaning up after SUCCESS\n" << std::endl;
        }
    }
    void halt() override
    {
        std::cout << name() <<": Halted." << std::endl;
        cleanup(true);
        // Do not forget to call this at the end.
        CoroActionNode::halt();
    }
};

#endif   // ACTION_NODES_H