//
//  ViewController.m
//  DeviceConnectUIAppTutorial
//
//  Copyright (c) 2014 NTT DOCOMO, INC.
//  Released under the MIT license
//  http://opensource.org/licenses/mit-license.php
//

#import "ViewController.h"
#import <DConnectSDK/DConnectSDK.h>

/*!
 セマフォのタイムアウト時間[秒]
 */
static int const _timeout = 500;


/*!
 セッションキー
 */
static NSString *const SESSION_KEY = @"SESSION_KEY";

@interface ViewController () {
    __weak IBOutlet UIButton *serviceDiscoveryBtn;
    __weak IBOutlet UIButton *requestAccessTokenBtn;
    __weak IBOutlet UIButton *sendNotificationBtn;
    __weak IBOutlet UIButton *deviceOrientationBtn;
    
    __weak IBOutlet UILabel *deviceInfo;
    __weak IBOutlet UILabel *accessTokenInfo;
    __weak IBOutlet UILabel *notificationInfo;
    __weak IBOutlet UILabel *deviceOrientationInfo;
    __weak IBOutlet UILabel *deviceOrientationEventInfo;
    
    DConnectArray *_services;
    NSString *_serviceId;
    NSString *_clientId;
    NSString *_accessToken;
}

@end

@implementation ViewController

- (void)viewDidLoad
{
    [super viewDidLoad];
	// Do any additional setup after loading the view, typically from a nib.
    
    _services = nil;
    _serviceId = nil;
    _clientId = nil;
    _accessToken = nil;
    
    [self displayInfo];

	// DConnectManagerの初期化
	DConnectManager *mgr = [DConnectManager sharedManager];
	// DConnectManagerの開始
	[mgr start];
//	// websocketサーバの起動
	[mgr startWebsocket];
	
	// nativeで作成する場合にはdelegateを設定すること
	mgr.delegate = self;
}

- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    // Dispose of any resources that can be recreated.
}

// delegate
- (void)manager:(DConnectManager *)manager didReceiveDConnectMessage:(DConnectMessage *)event {
    
    NSLog(@"receive delegate");
}

- (IBAction)touchUpNetworkServiceDiscovery:(UIButton *)sender {

    // スレッド起動
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{

        [self callServiceDiscoveryAPI];
    });
}
                   
/*!
 serviceDiscoveryを実行する
 @return YES 成功(_servicesに値格納)
 @return NO 失敗
 */
- (BOOL)callServiceDiscoveryAPI {
    
    _services = nil;
    __block BOOL result = NO;
    
    /* セマフォ準備 */
    dispatch_semaphore_t semaphore = dispatch_semaphore_create(0);
    dispatch_time_t timeout = dispatch_time(DISPATCH_TIME_NOW, NSEC_PER_SEC * _timeout);
    
	// serviceDiscoveryを実行する
    DConnectManager *mgr = [DConnectManager sharedManager];
	DConnectRequestMessage *request = [DConnectRequestMessage new];
	[request setAction: DConnectMessageActionTypeGet];
    [request setApi: DConnectMessageDefaultAPI];
	[request setProfile: DConnectServiceDiscoveryProfileName];
    [request setAccessToken:_accessToken];
    [request setString:[self packageName] forKey:DConnectMessageOrigin];
    
    NSLog(@"mgr request(service discovery API)");
	[mgr sendRequest: request callback:^(DConnectResponseMessage *response) {
        if (response != nil) {
            NSLog(@" - response is not null");
            NSLog(@" - response - result: %d", [response result]);
            if ([response result] == DConnectMessageResultTypeOk) {
                DConnectArray *services = [response arrayForKey: DConnectServiceDiscoveryProfileParamServices];
                NSLog(@" - response - services: %d", [services count]);
                
                _services = services;

                int serviceCount = [services count];
                for (int s = 0; s < serviceCount; s++) {
                    DConnectMessage *service = [services messageAtIndex: s];
                    
                    NSLog(@" - response - service[%d] -----", s);
                    
                    NSLog(@" --- id:%@", [service stringForKey: DConnectServiceDiscoveryProfileParamId]);
                    NSLog(@" --- name:%@", [service stringForKey: DConnectServiceDiscoveryProfileParamName]);
                    NSLog(@" --- type:%@", [service stringForKey: DConnectServiceDiscoveryProfileParamType]);
                    NSLog(@" --- online:%@", [service stringForKey: DConnectServiceDiscoveryProfileParamOnline]);
                    NSLog(@" --- config:%@", [service stringForKey: DConnectServiceDiscoveryProfileParamConfig]);
                }
                
                result = YES;
                
            } else {
                NSLog(@" - response - errorCode: %d", [response errorCode]);
            }
        }
        
        /* Wait解除 */
        dispatch_semaphore_signal(semaphore);
    }];
    
    /* 応答が返るまでWait */
    dispatch_semaphore_wait(semaphore, timeout);
    NSLog(@"mgr request(request(discovery API) finish");
    
    return result;
}


// アクセストークン要求
- (BOOL)callRequestAccessTokenAPI: (NSString *)serviceId {
    
    _accessToken = nil;
    __block BOOL result = NO;
    
    NSArray *scopes = [@[DConnectServiceDiscoveryProfileName,
                         DConnectDeviceOrientationProfileName,
                         DConnectNotificationProfileName]
                       sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];;

    
    /* セマフォ準備 */
    dispatch_semaphore_t semaphore = dispatch_semaphore_create(0);
    dispatch_time_t timeout = dispatch_time(DISPATCH_TIME_NOW, NSEC_PER_SEC * _timeout);
    
    [DConnectUtil asyncAuthorizeWithOrigin: [self packageName]
                                   appName: @"Test App"
                                    scopes: scopes
                                   success: ^(NSString *clientId, NSString *accessToken) {
                                       _clientId = clientId;
                                       _accessToken = accessToken;
                                       result = YES;
                                       NSLog(@" - response - accessToken: %@", accessToken);
                                       /* Wait解除 */
                                       dispatch_semaphore_signal(semaphore);

                                    }
                                     error:^(DConnectMessageErrorCodeType errorCode){
                                         NSLog(@" - response - errorCode: %d", errorCode);
                                         /* Wait解除 */
                                         dispatch_semaphore_signal(semaphore);

                                     }];

    /* 応答が返るまでWait */
    dispatch_semaphore_wait(semaphore, timeout);
    NSLog(@"mgr request(request_accesstoken API) finish");
    return result;
}




// Notificationを送信
- (BOOL)callNotificationAPI: (NSString *)serviceId {
    
    __block BOOL result = NO;
    
    /* セマフォ準備 */
    dispatch_semaphore_t semaphore = dispatch_semaphore_create(0);
    dispatch_time_t timeout = dispatch_time(DISPATCH_TIME_NOW, NSEC_PER_SEC * _timeout);

    // notifyを実行する
    DConnectRequestMessage *request = [DConnectRequestMessage new];
    [request setAction: DConnectMessageActionTypePost];
    [request setApi: DConnectMessageDefaultAPI];
    [request setProfile: DConnectNotificationProfileName];
    [request setAttribute: DConnectNotificationProfileAttrNotify];
    [request setServiceId: serviceId];
    [request setAccessToken: _accessToken];
    [request setString:[self packageName] forKey:DConnectMessageOrigin];
    
    [request setInteger: DConnectNotificationProfileNotificationTypePhone forKey: DConnectNotificationProfileParamType];
    [request setString: @"notification message" forKey: DConnectNotificationProfileParamBody];

    NSLog(@"mgr request(notification API)");
    DConnectManager *mgr = [DConnectManager sharedManager];
    [mgr sendRequest: request callback:^(DConnectResponseMessage *response) {
        NSLog(@"mgr request(callback)");
        if (response != nil) {
            NSLog(@" - response is not null");
            NSLog(@" - response - result: %d", [response result]);
            if ([response result] == DConnectMessageResultTypeOk) {
                NSString *notificationId = [response stringForKey: DConnectNotificationProfileParamNotificationId];
                NSLog(@" - response - OK notificationId: %@", notificationId);
                result = YES;
            } else {
                NSLog(@" - response - errorCode: %d", [response errorCode]);
                NSLog(@"message:%@", [response stringForKey:@"errorMessage"]);
                
            }
        }

        /* Wait解除 */
        dispatch_semaphore_signal(semaphore);
    }];
    
    /* 応答が返るまでWait */
    dispatch_semaphore_wait(semaphore, timeout);
    NSLog(@"mgr request(notification API) finish");
    return result;
}

// パッケージ名取得(bundleIdentifierを渡す)
- (NSString *)packageName {
    NSBundle *bundle = [NSBundle mainBundle];
    NSString *package = [bundle bundleIdentifier];
    return package;
}

/*!
 デバイス検索
 */
- (IBAction)onDeviceSearch:(id)sender {
    
    // スレッド起動
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{

        /* network_service_discoveryを実行しデバイスを1件選択する */
        BOOL result = [self callServiceDiscoveryAPI];
        
        NSString *message = nil;
        if (result == YES) {
            if (_services != nil && [_services count] > 0) {
                DConnectMessage *service = [_services messageAtIndex: 0];
                _serviceId = [service stringForKey: DConnectServiceDiscoveryProfileParamId];
                message = @"処理成功しました。";
            } else {
                message = @"serviceが見つかりませんでした。";
            }
        } else {
            message = @"タイムアウトしました。";
        }
        
        // 通知
        dispatch_async(dispatch_get_main_queue(), ^{
            if (_serviceId != nil) {
                [sendNotificationBtn setEnabled:YES];
                [deviceOrientationBtn setEnabled:YES];
                [deviceInfo setText: [NSString stringWithFormat: @"serviceId: %@", _serviceId]];
            } else {
                [deviceInfo setText: @""];
            }
            UIAlertView *alert =
                [[UIAlertView alloc]
                 initWithTitle: @"デバイス検索"
                 message: message
                 delegate:nil
                 cancelButtonTitle:nil
                 otherButtonTitles:@"OK", nil
                 ];
                [alert show];
        });
    });
}


/*!
 アクセストークン要求
 */
- (IBAction)onRequestAccessToken:(id)sender {
    
    // スレッド起動
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        // アクセストークン要求
        BOOL result = [self callRequestAccessTokenAPI: _serviceId];
        
        NSString *message = nil;
        if (result == YES) {
            message = @"処理成功しました。";
        } else {
            message = @"タイムアウトしました。";
        }
        
        // 通知
        dispatch_async(dispatch_get_main_queue(), ^{
            if (_accessToken != nil) {
                [serviceDiscoveryBtn setEnabled:YES];
                [accessTokenInfo setText: [NSString stringWithFormat: @"accessToken: %@", _accessToken]];
            } else {
                [accessTokenInfo setText: @""];
            }
            
            UIAlertView *alert =
            [[UIAlertView alloc]
             initWithTitle: @"アクセストークン要求"
             message: message
             delegate:nil
             cancelButtonTitle:nil
             otherButtonTitles:@"OK", nil
             ];
            [alert show];
        });
    });
}

/*!
 Notification送信
 */
- (IBAction)onSendNotification:(id)sender {
    
    // スレッド起動
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        // Notification送信
        BOOL result = [self callNotificationAPI:_serviceId];
        
        NSString *message = nil;
        if (result == YES) {
            message = @"処理成功しました。";
        } else {
            message = @"タイムアウトしました。";
        }
        
        // 通知
        dispatch_async(dispatch_get_main_queue(), ^{
            [notificationInfo setText: [NSString stringWithFormat: @"notification: %@", message]];
            
            UIAlertView *alert =
            [[UIAlertView alloc]
             initWithTitle: @"Notification送信"
             message: message
             delegate:nil
             cancelButtonTitle:nil
             otherButtonTitles:@"OK", nil
             ];
            [alert show];
        });
    });
}

static DConnectEventHelper *_helper = nil;

/*!
 DeviceOrientationイベント
 */
- (IBAction)onDeviceOrientaionEvent:(id)sender {
   
    // DeviceOrientationイベント登録リクエスト
    DConnectRequestMessage *request = [DConnectRequestMessage new];
    [request setAction: DConnectMessageActionTypePut];
    [request setApi: DConnectMessageDefaultAPI];
    [request setServiceId: _serviceId];
    [request setProfile: DConnectDeviceOrientationProfileName];
    [request setAttribute: DConnectDeviceOrientationProfileAttrOnDeviceOrientation];
    [request setSessionKey: SESSION_KEY];
    [request setAccessToken:_accessToken];
    [request setString:[self packageName] forKey:DConnectMessageOrigin];
    
    
    [[DConnectEventHelper sharedHelper] registerEventWithRequest: request
                     responseHandler: ^(DConnectResponseMessage *response)
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            NSString *message = nil;
            if ([response result] == DConnectMessageResultTypeOk) {
                message = @"処理成功しました。";
            } else {
                message = @"処理失敗しました。";
            }
            [deviceOrientationInfo setText: [NSString stringWithFormat: @"event: %@", message]];
        });
    }
                      messageHandler:^(DConnectMessage *message)
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            
            DConnectMessage *orientation = [message messageForKey: DConnectDeviceOrientationProfileParamOrientation];
            
            DConnectMessage *acceleration = [orientation  messageForKey: DConnectDeviceOrientationProfileParamAcceleration];

            double x = [acceleration doubleForKey: DConnectDeviceOrientationProfileParamX];
            double y = [acceleration doubleForKey: DConnectDeviceOrientationProfileParamY];
            double z = [acceleration doubleForKey: DConnectDeviceOrientationProfileParamZ];
            
            [deviceOrientationEventInfo setText: [NSString stringWithFormat: @"acceleration: x=%.1lf, y=%.1lf, z=%.1lf", x, y, z]];
            
        });
    }];
}

/*!
 情報表示
 */
- (void) displayInfo {
    
    if (_serviceId != nil) {
        [deviceInfo setText: _serviceId];
    } else {
        [deviceInfo setText: @""];
    }
    
    if (_accessToken != nil) {
        [accessTokenInfo setText: _accessToken];
    } else {
        [accessTokenInfo setText: @""];
    }
    
    [notificationInfo setText: @""];
}


@end
