#include <Arduino.h>
#include <ArduinoOTA.h>

#ifdef ESP8266 // if ESP8266
#include <BlynkSimpleEsp8266_SSL.h>
#include <ESP8266WiFi.h>
#endif

#ifdef ESP32
//#include <BlynkSimpleEsp32.h>
#include <BlynkSimpleEsp32_SSL.h>
#include <WiFi.h>
#define LED_BUILTIN 2
#endif

char auth[] = "x";
const char *hostname = "esp-doorbell";

char auth_entrance[] = "x"; // Auth key for entrance gate //XXX

#include "../../common/lib/common.cpp"

#define PIN_DOORBELL 14   // D5, GPIO14
#define PIN_CHIME 4       // D2, GPIO4
#define PIN_DOOR_SENSOR 5 // D1

//#define BACKEND_HOST "192.168.0.101" // local
#define BACKEND_HOST "x"                                                                          // running on EC2
#define URL_VIDEO "rtsp://x/21"                                                    // Camera video stream
#define URL_NA "https://image.freepik.com/free-icon/not-available-abbreviation-inside-a-circle_318-33662.jpg" // N/A placeholder image
#define DELAY_AUTO_PRESS 4000L                                                                                // How long to wait between doorbell press and opening the gate
#define DELAY_IGNORE_HW_LIGHT 5000L                                                                           // How long to ignore hw light after bootup and silence changes
#define DELAY_LIGHTS_FLASHING 10000L                                                                          // How long should we ignore the flashing LEDs until the next doorbell ring
#define DELAY_BUTTON_HOLD 60000L                                                                              // Keep the recent doorbell ring button state for 60 seconds
#define EMAIL_ON_RING true                                                                                    // should we email a photo of the front camera? requires flask util service (in ~/bin)

#define V_DOORBELL_RUNG V1
#define V_CHIME V2
#define V_URL_VIDEO V3
#define V_AUTO_OPEN V4
#define V_IMAGE V5
#define V_TIMESTAMP V6
#define V_NOTIFY V7
#define V_ENTRANCE_BUTTON V3 // Note this is on a different device

WidgetBridge bridge_entrance(V_AUTO_OPEN);

bool doorbell_recently_pressed = false;
bool doorbell_visual_state = false;
bool door_sensor_already_pressed = false;
bool door_sensor_pressed_recently = false;
bool auto_open = false;
bool send_notification = true;

void reset_recently_pressed_state() {
    log_line("Already pressed state restored");

    doorbell_recently_pressed = false;
}

void reset_recently_pressed_button() {
    log_line("Pressed button state restored");

    doorbell_visual_state = false;

    Blynk.virtualWrite(V_DOORBELL_RUNG, doorbell_visual_state);
}

void send_doorbell_state() {
    Blynk.virtualWrite(V_DOORBELL_RUNG, doorbell_visual_state); // Reset the app button in case it was stuck
}

void door_sensor_pushed() {

    if (send_notification) {
        log_line("Door sensor notified");

        String notification = "⚠️ Someone opened one of the back doors!";
        Blynk.notify(notification);
    }
}

void capture_and_send_image() {
    String filename;

    // capture an image of the front gate and email on ring
    if (EMAIL_ON_RING) {
        log_line("Capturing image on backend...");

        WiFiClient client;
        if (client.connect(BACKEND_HOST, 5000)) { // TODO should be a proper hostname
            client.print("GET /x HTTP/1.1\r\nHost: " + String(BACKEND_HOST) + ":5000\r\nConnection: close\r\n\r\n");

            while (client.connected() || client.available()) {
                if (client.available()) {
                    String response = client.readStringUntil('!');

                    log_line("Backend response: " + response);

                    int pos_date_start = response.indexOf(">");
                    int pos_date_end = response.indexOf("<");
                    filename = response.substring(pos_date_start + 1, pos_date_end);
                }
            }
            client.stop();
        }

        log_line("Filename: " + filename);

        String url = "https://s3-us-west-1.amazonaws.com/x/captures/" + filename + ".jpg";

        Blynk.setProperty(V_IMAGE, "urls", url);
        Blynk.virtualWrite(V_IMAGE, 1);

        Blynk.virtualWrite(V8, url);

        String timestamp = get_timestamp();
        Blynk.virtualWrite(V_TIMESTAMP, "Captured at " + timestamp);
    }
}

void doorbell_pushed() {
    if (send_notification) {
        log_line("Doorbell notified");

        String notification = "👋🏻 Someone is ringing the doorbell! Check your email for photo.";

        if (auto_open) {
            notification = notification + " Auto-opening the entrance in " + String(DELAY_AUTO_PRESS / 1000) + " seconds.";
        }

        Blynk.notify(notification);
    }

    send_doorbell_state();

    blynk_timer.setTimeout(DELAY_LIGHTS_FLASHING, reset_recently_pressed_state); // Reset button state
    blynk_timer.setTimeout(DELAY_BUTTON_HOLD, reset_recently_pressed_button);    // Reset visual state in app

    // open the gate if the Auto Open setting is on
    if (auto_open) {
        blynk_timer.setTimeout(DELAY_AUTO_PRESS, []() {
            log_line("Auto-opening gate");

            bridge_entrance.virtualWrite(V_ENTRANCE_BUTTON, 1);
        });
    }

    capture_and_send_image();
}

/*
void ICACHE_RAM_ATTR handle_door_sensor() {
    door_sensor_pressed_recently = true;

    if (door_sensor_already_pressed == false) {
        door_sensor_already_pressed = true;

        blynk_timer.setTimeout(50L, door_sensor_pushed);
    }
}
*/

void ICACHE_RAM_ATTR handle_doorbell_push() {
    doorbell_visual_state = true;

    if (doorbell_recently_pressed == false) {
        doorbell_recently_pressed = true;

        blynk_timer.setTimeout(50L, doorbell_pushed);
    }
}

BLYNK_WRITE(V_NOTIFY) {
    send_notification = (param.asInt() == 1);

    log_line("Notification from param " + String(param.asInt()) + " set to " + String(send_notification ? "on" : "off"));
}

BLYNK_WRITE(V_AUTO_OPEN) {
    auto_open = (param.asInt() == 1);

    log_line("Auto-open set to " + String(auto_open ? "on" : "off"));
}

BLYNK_CONNECTED() {
    blynk_connected();

    bridge_entrance.setAuthToken(auth_entrance);

    send_doorbell_state();
}

BLYNK_APP_CONNECTED() {
    Blynk.virtualWrite(V_DOORBELL_RUNG, doorbell_visual_state);

    Blynk.syncAll(); // To sync doorbell silent state with hardware

    // This resets the video stream buffer to receive fresh video
    Blynk.setProperty(V_URL_VIDEO, "url", URL_VIDEO);

    log_line("Blynk reconnected.");
}

void init_pins() {
    pinMode(PIN_DOORBELL, INPUT);
    attachInterrupt(digitalPinToInterrupt(PIN_DOORBELL), handle_doorbell_push, RISING);

    Blynk.virtualWrite(V_DOORBELL_RUNG, doorbell_visual_state);
}

void setup() {
    init_pins();

    init_common();

    log_line("Ready.");
}
