package org.springframework.boot.showcase;

import java.util.Collection;
import java.util.Objects;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.boot.context.properties.ConfigurationPropertiesBindHandlerAdvisor;
import org.springframework.boot.context.properties.bind.AbstractBindHandler;
import org.springframework.boot.context.properties.bind.BindContext;
import org.springframework.boot.context.properties.bind.BindHandler;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.PlaceholdersResolver;
import org.springframework.boot.context.properties.bind.PropertySourcesPlaceholdersResolver;
import org.springframework.boot.context.properties.source.ConfigurationProperty;
import org.springframework.boot.context.properties.source.ConfigurationPropertyName;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Role;
import org.springframework.context.annotation.Scope;
import org.springframework.core.convert.ConversionException;
import org.springframework.core.env.Environment;

import com.github.openjson.JSONException;
import com.github.openjson.JSONObject;

/**
 * This Configuration enables the definition of vault-attributes as Values for ConfigurationProperties.
 *
 * You have to define in application.yml or any other property-file the vault-attribute as follows:
 * <PRE>secret-spec{''safe':'SAFENAME','aimUser':'AIMUSER','objectName':'OBJECTNAME', 'name':'NAME'}}</PRE>
 *
 * The {@link VaultPropertiesAwareBindHandler} just interprets this values and reads the corresponding value from the vault.
 * The Spring-Binder for ConfigurationProperties then uses this value from the vault.
 *
 * Format:
 * <OL>
 *     <LI>safe: Name of the safe</LI>
 *     <LI>aimUser: user (application-name)</LI>
 *     <LI>objectname: Name of the vault-object</LI>
 *     <LI>name: name of the attribute of the vault-object. Valid values: "User", "Content", "Comment", "Url"</LI>
 * </OL>
 */
@Configuration
public class VaultConfigurationPropertiesConfiguration {

	@Bean
	@Scope(BeanDefinition.SCOPE_SINGLETON)
	@Role(BeanDefinition.ROLE_INFRASTRUCTURE)
	public ConfigurationPropertiesBindHandlerAdvisor vaultPropertiesBindHandlerAdvisor(final IVault vault, final Environment environment) {
		return (h) -> h!= null ? new VaultPropertiesAwareBindHandler(h, vault, environment) : null;
	}

	@Bean
	@Scope(BeanDefinition.SCOPE_SINGLETON)
	@Role(BeanDefinition.ROLE_INFRASTRUCTURE)
	public IVault vault() {
		return new IVault.DummyVault(); // only a dummy here: proprietary library in reality.
	}

	static class VaultPropertiesAwareBindHandler extends AbstractBindHandler {
		private static final Logger LOGGER = LoggerFactory.getLogger(VaultPropertiesAwareBindHandler.class);
		private static final String VAULT_PREFIX ="secret-spec";
		private static final Collection<String> VALID_NAME_VALUES = Set.of("User", "Content", "Comment", "Url");

		private final IVault vault;

		private final PlaceholdersResolver placeholdersResolver;

		public VaultPropertiesAwareBindHandler(final BindHandler bindHandler, final IVault vault, final Environment environment) {
			super(bindHandler);
			this.vault = vault;
			this.placeholdersResolver = new PropertySourcesPlaceholdersResolver(environment);
		}

		@Override
		public Object onFailure(final ConfigurationPropertyName name, final Bindable<?> target, final BindContext context, final Exception error) throws Exception {
			final Object result;
			if (error instanceof ConversionException) {
				final ConfigurationProperty configurationProperty = context.getConfigurationProperty();
				final Object object = configurationProperty != null ? configurationProperty.getValue() : null;
				final Object baseResult = placeholdersResolver.resolvePlaceholders(object);

				final String trimmedValue = baseResult instanceof String ? baseResult.toString().trim() : null;
				final boolean vaultAttributeSpecCandidate = trimmedValue != null && trimmedValue.startsWith(VAULT_PREFIX + "{") && trimmedValue.endsWith("}");
				if (vaultAttributeSpecCandidate) {
					final String vaultAttribute = determineVaultAttribute(trimmedValue);
					result = vaultAttribute != null ? vaultAttribute : super.onFailure(name, target, context, error); // default von super.onFailure: throw error ;-)
				} else {
					result = super.onFailure(name, target, context, error); // default von super.onFailure: throw error ;-)
				}
			} else {
				result = super.onFailure(name, target, context, error); // default von super.onFailure: throw error ;-)
			}
			return result;
		}

		@Override
		public Object onSuccess(final ConfigurationPropertyName configurationPropertyName, final Bindable<?> target, final BindContext context, final Object value) {
			final Object result;
			final String trimmedValue = value instanceof String ? value.toString().trim() : null;
			final boolean vaultSpecCandidate = trimmedValue != null && trimmedValue.startsWith(VAULT_PREFIX + "{") && trimmedValue.endsWith("}");
			if (vaultSpecCandidate) {
				final String vaultAttribute = determineVaultAttribute(trimmedValue);
				result = vaultAttribute != null ? vaultAttribute : value;
			} else {
				result = value;
			}

			return super.onSuccess(configurationPropertyName, target, context, result);
		}

		private String determineVaultAttribute(final String vaultAttributeSpec) {
			String result = null;
			final String jsonCandicate = vaultAttributeSpec.substring(VAULT_PREFIX.length());
			try {
				final JSONObject jsonObject = new JSONObject(jsonCandicate);
				final String safe = jsonObject.optString("safe", "").trim();
				final String aimUser = jsonObject.optString("aimUser", "").trim();
				final String objectName = jsonObject.optString("objectName", "").trim();
				final boolean safeDefined = !Objects.equals(safe, "");
				final boolean aimUserDefined = !Objects.equals(aimUser, "");
				final boolean objectNameDefined = !Objects.equals(objectName, "");
				final boolean nameDefined = !Objects.equals(jsonObject.optString("name", "").trim(), "");
				final int keys = jsonObject.length();
				final boolean valid = safeDefined && aimUserDefined && objectNameDefined && (nameDefined || keys == 3); // name is optional, default is "Content" :-)
				if (!valid) {
					LOGGER.warn("Invalid Vault-Attribute-Spec: {}. Please check.", vaultAttributeSpec);
				} else {
					 final String name = jsonObject.optString("name", "Content").trim();
					 if (VALID_NAME_VALUES.contains(name)) {
						 result = vault.get(safe, aimUser, objectName, name);
						 if (result != null) {
							 LOGGER.info("Vault-Attribute retrieved for {}, {}, {}, {}", safe, aimUser, objectName, name);
						 } else {
							 LOGGER.warn("NO VALUE FOR Vault-Attribute-Spec {} retrieved.", vaultAttributeSpec);
						 }
					 } else {
						 LOGGER.warn("Invalid Vault-Attribute-Spec: {}. Value of attribute 'name' must be one of {}", vaultAttributeSpec, VALID_NAME_VALUES);
					 }
				}
			} catch (final JSONException jsonException) {
				LOGGER.info("Secret-spec not followed by valid JSON. Potential configuration error: {}", vaultAttributeSpec);
			} catch (final Exception exception) {
				LOGGER.info("Could not read attribute from Vault {}", exception);
			}
			return result;
		}
	}
}
