package com.example.demo;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.server.LocalServerPort;
import org.springframework.http.HttpStatus;
import org.springframework.test.web.reactive.server.WebTestClient;

@SpringBootTest(
    webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT,
    properties = {"management.endpoint.health.group.readiness.include=my"}
)
class ActuatorTests {

    @LocalServerPort
    private int port;

    WebTestClient client;

    @BeforeEach
    void setUp() {
        client = WebTestClient.bindToServer().baseUrl("http://localhost:" + port).build();
    }

    /**
     * This is due to the myHealthIndicator bean defined in MyConfig.java.
     */
    @Test
    void healthShouldBeDown() {
        client.get()
            .uri("/actuator/health")
            .exchange()
            .expectStatus().isEqualTo(HttpStatus.SERVICE_UNAVAILABLE);
    }

    /**
     * This is because we set:
     * management.endpoint.health.group.readiness.include=my
     * <p>
     * Since that health indicator is down, we expect readiness to be down too.
     */
    @Test
    void readinessShouldBeDownDueToMyHealthIndicator() {
        client.get()
            .uri("/actuator/health/readiness")
            .exchange()
            .expectStatus().isEqualTo(HttpStatus.SERVICE_UNAVAILABLE);
    }

    /**
     * We haven't registered this yet.
     */
    @Test
    void additionalReadinessPathShouldNotBePresent() {
        client.get()
            .uri("/readyz")
            .exchange()
            .expectStatus().isNotFound();
    }

    /**
     * This is because we are not including myHealthIndicator in liveness group.
     * <p>
     * So, even though myHealthIndicator is down, liveness should be up.
     */
    @Test
    void livenessShouldBeUp() {
        client.get()
            .uri("/actuator/health/liveness")
            .exchange()
            .expectStatus().isOk();
    }

    /**
     * We haven't registered this yet.
     */
    @Test
    void additionalLivenessPathShouldNotBePresent() {
        client.get()
            .uri("/livez")
            .exchange()
            .expectStatus().isNotFound();
    }
}
