package com.example.demo;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.web.server.LocalServerPort;
import org.springframework.http.HttpStatus;
import org.springframework.test.web.reactive.server.WebTestClient;

@SpringBootTest(
    webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT,
    properties = {"management.endpoint.health.group.readiness.additional-path=server:/readyz"}
)
class ActuatorTestsWithAdditionalReadinessPathTests {

    @LocalServerPort
    private int port;

    WebTestClient client;

    @BeforeEach
    void setUp() {
        client = WebTestClient.bindToServer().baseUrl("http://localhost:" + port).build();
    }

    /**
     * This is due to the myHealthIndicator bean defined in MyConfig.java.
     */
    @Test
    void healthShouldBeDown() {
        client.get()
            .uri("/actuator/health")
            .exchange()
            .expectStatus().isEqualTo(HttpStatus.SERVICE_UNAVAILABLE);
    }

    /**
     * This is because we are not including myHealthIndicator in readiness group.
     * <p>
     * So, even though myHealthIndicator is down, readiness should be up.
     */
    @ParameterizedTest
    @CsvSource({"/actuator/health/readiness", "/readyz"})
    void readinessShouldBeDownDueToMyHealthIndicator(final String uri) {
        client.get()
            .uri(uri)
            .exchange()
            .expectStatus().isOk();
    }

    /**
     * This is because we are not including myHealthIndicator in liveness group.
     * <p>
     * So, even though myHealthIndicator is down, liveness should be up.
     */
    @Test
    void livenessShouldBeUp() {
        client.get()
            .uri("/actuator/health/liveness")
            .exchange()
            .expectStatus().isOk();
    }

    /**
     * We haven't registered this yet.
     */
    @Test
    void additionalLivenessPathShouldNotBePresent() {
        client.get()
            .uri("/livez")
            .exchange()
            .expectStatus().isNotFound();
    }
}
