package org.example.example;

import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.listener.KeyspaceEventMessageListener;
import org.springframework.data.redis.listener.PatternTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.listener.Topic;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

@Slf4j
public class RedisEventSubscriberBug extends KeyspaceEventMessageListener implements AutoCloseable {

    interface ValueReader{

        String read();
    }

    private String key;

    private final Lock lock = new ReentrantLock();
    private final Condition condition = lock.newCondition();

    public RedisEventSubscriberBug(RedisMessageListenerContainer listenerContainer, String key) {
        super(listenerContainer);
        this.key = key;
        super.init();
    }

    @Override
    protected void doRegister(RedisMessageListenerContainer container) {
        var keyEvent = createTopicHelper(this.key);
        container.addMessageListener(this, keyEvent);
    }

    @Override
    protected void doHandleMessage(Message message) {
        log.info("0001; Redis event received: {} for key: {}", message.toString(), this.key);
        lock.lock();
        try {
            condition.signal();
        } finally {
            lock.unlock();
        }
    }

    private static Topic createTopicHelper(String key) {
        return new PatternTopic("__keyspace@*__:" + key);
    }

    public String awaitRedisEvent(ValueReader valueReader, long eventWaitingTime) {
        return awaitEvent(valueReader, eventWaitingTime, false);
    }

    public String awaitRedisEmptyEvent(ValueReader valueReader, long eventWaitingTime) {
        return awaitEvent(valueReader, eventWaitingTime, true);
    }

    private String awaitEvent(ValueReader valueReader, long eventWaitingTime, boolean waitForEmptyValue) {
        String value = null;
        lock.lock();
        try {
            while ((waitForEmptyValue ? (value = valueReader.read()) != null : (value = valueReader.read()) == null)
                    && (eventWaitingTime > 0)) {
                log.debug("0002;waiting for event for {} ms", eventWaitingTime);
                long startTime = System.currentTimeMillis();
                boolean waitSuccesfull = condition.await(eventWaitingTime, TimeUnit.MILLISECONDS);
                long endTime = System.currentTimeMillis();
                eventWaitingTime -= (endTime - startTime);
                log.debug("0005;rechecking after {}", waitSuccesfull ? "event" : "timeout");
                // read even after timeout, pub/sub is not 100% reliable
            }
        } catch (InterruptedException e) {
            log.warn("0003;await interrupted due to {}", e.getMessage());
            Thread.currentThread().interrupt();
        } finally {
            lock.unlock();
        }
        log.info("0004;value returned from redis after awaitEvent '{}'", value);
        return value;
    }

    @Override
    public void close() throws Exception {
        destroy();
    }
}
