package org.example.example;

import com.github.noconnor.junitperf.JUnitPerfInterceptor;
import com.github.noconnor.junitperf.JUnitPerfReportingConfig;
import com.github.noconnor.junitperf.JUnitPerfTest;
import com.github.noconnor.junitperf.JUnitPerfTestActiveConfig;
import com.github.noconnor.junitperf.reporting.providers.HtmlReportGenerator;
import lombok.extern.slf4j.Slf4j;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.data.redis.listener.PatternTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.listener.adapter.MessageListenerAdapter;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = { SubmitBugTestCase.Config.class })
@Slf4j
@ExtendWith(JUnitPerfInterceptor.class)
public class SubmitBugTestCase{

    @Configuration
    public static class Config{

        private List<String> sentinelnodes = new ArrayList<String>(
                Arrays.asList("172.23.0.4:6000", "172.23.0.4:6001", "172.23.0.4:6002"));
        private String mastername = "tesaredis";

        @Bean
        public LettuceConnectionFactory lettuceSentinelConnectionFactory() {
            RedisSentinelConfiguration sentinelConfig = new RedisSentinelConfiguration(mastername, new HashSet(sentinelnodes));
            return new LettuceConnectionFactory(sentinelConfig);
        }

        @Bean
        public StringRedisTemplate stringRedisTemplate(LettuceConnectionFactory lettuceSentinelConnectionFactory) {
            StringRedisTemplate redisTemplate = new StringRedisTemplate();
            redisTemplate.setConnectionFactory(lettuceSentinelConnectionFactory);
            return redisTemplate;
        }

        @Bean
        public TaskScheduler taskScheduler() {
            return new ThreadPoolTaskScheduler();
        }

        @Bean
        MessageListenerAdapter listenerAdapter() {
            return new MessageListenerAdapter();
        }

        @Bean
        public RedisMessageListenerContainer redisMessageListenerContainer(
                LettuceConnectionFactory lettuceSentinelConnectionFactory, MessageListenerAdapter listenerAdapter) {
            var redisMessageListenerContainer = new RedisMessageListenerContainer();
            redisMessageListenerContainer.setConnectionFactory(lettuceSentinelConnectionFactory);
            redisMessageListenerContainer.setTaskExecutor(Executors.newFixedThreadPool(5));
            // TODO remove the hack if the redis issue is solved see
//            redisMessageListenerContainer.addMessageListener(listenerAdapter, new PatternTopic("Dummy"));
            return redisMessageListenerContainer;
        }

    }

    public static final HtmlReportGenerator HTML_REPORTER = new HtmlReportGenerator("target/junitperf-report/index.html");

    @Autowired
    private StringRedisTemplate stringRedisSentinelTemplate;

    @Autowired
    private RedisMessageListenerContainer redisMessageListenerContainer;

    private int keyEventExpiryTimeMilliSeconds = 300;

    @JUnitPerfTestActiveConfig
    private static final JUnitPerfReportingConfig PERF_CONFIG = JUnitPerfReportingConfig.builder().reportGenerator(HTML_REPORTER)
            .build();

    @Test
    @JUnitPerfTest(threads = 4, durationMs = 4000)
    public void testRedisPubSub() {
        var testValue = createTestValue();
        String key = createNewEventKey(testValue);

        try (RedisEventSubscriberBug listener = createEventSubscriber(key)) {
            new Thread(() -> sendNotification(testValue, 100)).start();
            var iccidNew = listener.awaitRedisEvent(() -> retrieveNewEvent(testValue),  keyEventExpiryTimeMilliSeconds);
        } catch (Exception e) {
            log.error("1001; failure: {}", e.getMessage());
            throw new RuntimeException("Test Failed:", e);
        }

        // avoid exception if the tests are done but last threads are still running
        try {
            TimeUnit.MILLISECONDS.sleep(500);
        } catch (InterruptedException e) {
            log.info("1002;Interrupted because test end: {}", e.getMessage());
        }
    }

    private void sendNotification(String key, int afterTimeout) {
        try {
            TimeUnit.MILLISECONDS.sleep(afterTimeout);
        } catch (InterruptedException e) {
            log.info("1003;Interrupted because test end: {}", e.getMessage());
        }

        storeNewEvent(key, "Test:" + key);
    }

    public void storeNewEvent(String iccidOld, String iccidNew) {
        String key = createNewEventKey(iccidOld);
        storeKeyValuePairWithTtl(key, iccidNew, keyEventExpiryTimeMilliSeconds, TimeUnit.MILLISECONDS);
        log.info("1004;Storing new event key {}, value '{}', expiry {}", key, iccidNew, keyEventExpiryTimeMilliSeconds);
    }

    public String retrieveNewEvent(String key) {
        ValueOperations<String, String> values = stringRedisSentinelTemplate.opsForValue();
        return values.get(createNewEventKey(key));
    }

    private String createNewEventKey(String key) {
        return "NEW_EVENT:" + key;
    }


    private void storeKeyValuePairWithTtl(String key, String value, long timeout, TimeUnit unit) {
        ValueOperations<String, String> values = stringRedisSentinelTemplate.opsForValue();
        values.set(key, value, timeout, unit);
    }


    private RedisEventSubscriberBug createEventSubscriber(String key) {
        return new RedisEventSubscriberBug(redisMessageListenerContainer, key);
    }

    // for unique values
    private AtomicLong counter = new AtomicLong(1);
    private final int max = (int) Math.pow(10, 5);

    private String createTestValue() {
        return String.format("7838020000%05d", Math.floorMod(counter.getAndIncrement(), max));
    }

}
