﻿/*

 pxCore Copyright 2005-2018 John Robinson

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

*/

// pxBrowserExample.cpp

#include "pxCore.h"
#include "pxTimer.h"
#include "pxEventLoop.h"
#include "pxWindow.h"

#define ANIMATION_ROTATE_XYZ
#include "pxContext.h"
#include "pxScene2d.h"




pxEventLoop  eventLoop;

pxContext context;

#include "pxIView.h"

class pxBrowserView: public pxIView
{
public:
    pxBrowserView() : mWidth(0), mHeight(0), mRefCount(0) {}
    virtual unsigned long AddRef() {
      return rtAtomicInc(&mRefCount);
    }

    virtual unsigned long Release() {
      long l = rtAtomicDec(&mRefCount);
      if (l == 0) delete this;
      return l;
    }
    virtual void RT_STDCALL onSize(int32_t w, int32_t h){mWidth = w; mHeight = h;}

    // events return true if the event was consumed by the view
    virtual bool RT_STDCALL onMouseDown(int32_t x, int32_t y, uint32_t flags) {return true;}
    virtual bool RT_STDCALL onMouseUp(int32_t x, int32_t y, uint32_t flags) {return true;}
    virtual bool RT_STDCALL onMouseMove(int32_t x, int32_t y) {return true;}

    virtual bool RT_STDCALL onScrollWheel(float dx, float dy) { UNUSED_PARAM(dx); UNUSED_PARAM(dy); return false; };

    virtual bool RT_STDCALL onMouseEnter() {return true;}
    virtual bool RT_STDCALL onMouseLeave() {return true;}

    virtual bool RT_STDCALL onFocus() {return true;}
    virtual bool RT_STDCALL onBlur() {return true;}

    virtual bool RT_STDCALL onKeyDown(uint32_t keycode, uint32_t flags) {return true;}
    virtual bool RT_STDCALL onKeyUp(uint32_t keycode, uint32_t flags) {return true;}
    virtual bool RT_STDCALL onChar(uint32_t codepoint) {return true;}

    virtual void RT_STDCALL onUpdate(double t) {}
    virtual void RT_STDCALL onDraw()
    {
      float color[4] = {1.0, 0.0, 0.0, 1.0};
      context.drawRect(mWidth, mHeight, 0, color, color);
    }

    virtual void RT_STDCALL setViewContainer(pxIViewContainer* l) {}
    virtual void RT_STDCALL onCloseRequest() {};
protected:
    int mWidth;
    int mHeight;
    rtAtomic mRefCount;
};

typedef rtRef<pxBrowserView> pxBrowserViewRef;


class pxBrowserViewContainer: public pxIViewContainer
{
public:
    pxBrowserViewContainer() : mRefCount(0), mBrowserView(NULL) {}

    virtual unsigned long AddRef() {
      return rtAtomicInc(&mRefCount);
    }

    virtual unsigned long Release() {
      long l = rtAtomicDec(&mRefCount);
      if (l == 0) delete this;
      return l;
    }

    virtual void RT_STDCALL invalidateRect(pxRect* /*r*/) {}
    virtual void* RT_STDCALL getInterface(const char* /*t*/) { return NULL;}

    rtError setView(pxBrowserView* v)
    {
      mBrowserView = v;
      return RT_OK;
    }

    void onSize(int32_t w, int32_t h)
    {
      if (mBrowserView)
        mBrowserView->onSize(w, h);
    }

    void onMouseDown(int32_t x, int32_t y, uint32_t flags)
    {
      if (mBrowserView)
        mBrowserView->onMouseDown(x, y, flags);
    }

    void onMouseUp(int32_t x, int32_t y, uint32_t flags)
    {
      if (mBrowserView)
        mBrowserView->onMouseUp(x, y, flags);
    }

    void onMouseLeave()
    {
      if (mBrowserView)
        mBrowserView->onMouseLeave();
    }

    void onMouseMove(int32_t x, int32_t y)
    {
      if (mBrowserView)
        mBrowserView->onMouseMove(x, y);
    }

    void onScrollWheel(float dx, float dy)
    {
      if (mBrowserView)
        mBrowserView->onScrollWheel(dx, dy);
    }

    void onFocus()
    {
      if (mBrowserView)
        mBrowserView->onFocus();
    }

    void onBlur()
    {
      if (mBrowserView)
        mBrowserView->onBlur();
    }

    void onKeyDown(uint32_t keycode, uint32_t flags)
    {
      if (mBrowserView)
        mBrowserView->onKeyDown(keycode, flags);
    }

    void onKeyUp(uint32_t keycode, uint32_t flags)
    {
      if (mBrowserView)
        mBrowserView->onKeyUp(keycode, flags);
    }

    void onChar(uint32_t c)
    {
      if (mBrowserView)
        mBrowserView->onChar(c);
    }

    void onDraw()
    {
      if (mBrowserView)
        mBrowserView->onDraw();
    }

    void onUpdate(double t)
    {
      if (mBrowserView)
        mBrowserView->onUpdate(t);
    }

protected:
    rtAtomic mRefCount;
    pxBrowserViewRef mBrowserView;
};

typedef rtRef<pxBrowserViewContainer> pxBrowserViewContainerRef;




class sceneWindow : public pxWindow, public pxIViewContainer
{
public:
  sceneWindow(): mWidth(0), mHeight(0), mBrowserViewContainer(NULL) {}
  virtual ~sceneWindow()
  {
    mBrowserViewContainer = NULL;
  }

  void init(int x, int y, int w, int h)
  {
    pxWindow::init(x,y,w,h);
    mWidth = w;
    mHeight = h;
    mBrowserViewContainer = new pxBrowserViewContainer();
    mBrowserViewContainer->setView( new pxBrowserView());
    mBrowserViewContainer->onSize(w,h);
  }

  void* getInterface(const char* /*name*/)
  {
     return NULL;
  }
  
  virtual void invalidateRect(pxRect* r)
  {
    pxWindow::invalidateRect(r);
  }

  void close()
  {
    onCloseRequest();
  }
protected:

  virtual void onSize(int32_t w, int32_t h)
  {
    if (mWidth != w || mHeight != h)
    {
      mWidth  = w;
      mHeight = h;
      if (mBrowserViewContainer)
        mBrowserViewContainer->onSize(w, h);
    }
  }

  virtual void onMouseDown(int32_t x, int32_t y, uint32_t flags)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onMouseDown(x, y, flags);
  }

  virtual void onCloseRequest()
  {
    context.term();
    eventLoop.exit();
  }

  virtual void onMouseUp(int32_t x, int32_t y, uint32_t flags)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onMouseUp(x, y, flags);
  }

  virtual void onMouseLeave()
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onMouseLeave();
  }

  virtual void onMouseMove(int32_t x, int32_t y)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onMouseMove(x, y);
  }

  virtual void onScrollWheel(float dx, float dy)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onScrollWheel(dx, dy);
  }

  virtual void onFocus()
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onFocus();
  }
  virtual void onBlur()
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onBlur();
  }

  virtual void onKeyDown(uint32_t keycode, uint32_t flags)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onKeyDown(keycode, flags);
  }

  virtual void onKeyUp(uint32_t keycode, uint32_t flags)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onKeyUp(keycode, flags);
  }

  virtual void onChar(uint32_t c)
  {
    if (mBrowserViewContainer)
      mBrowserViewContainer->onChar(c);
  }

  virtual void onDraw(pxSurfaceNative )
  {
    context.setSize(mWidth, mHeight);
    context.clear(mWidth, mHeight);
    if (mBrowserViewContainer)
      mBrowserViewContainer->onDraw();
  }

  virtual void onAnimationTimer()
  {
    if (mBrowserViewContainer)
    {
      mBrowserViewContainer->onUpdate(pxSeconds());
    }
  }

protected:
  int32_t mWidth;
  int32_t mHeight;

  pxBrowserViewContainerRef mBrowserViewContainer;
};
sceneWindow win;


int pxMain(int argc, char* argv[])
{

  win.init(10, 10, 1280, 720);
  win.setVisibility(true);

  win.setAnimationFPS(60);


  context.init();
  eventLoop.run();

  return 0;
}
