#!/usr/bin/env python
import argparse
import time

from confluent_kafka import Producer
from confluent_kafka.admin import AdminClient, NewTopic
from confluent_kafka.error import KafkaError

# Use broker:9092 if running within a Docker container, else "127.0.0.1:9092"

KAFKA_BROKER_ADDR = "127.0.0.1:9092"
print(f"{KAFKA_BROKER_ADDR=}")


def create_topics(
    topic_names: list[str],
    broker_addr: str,
    num_partitions=1,
    replication_factor=1,
) -> None:
    """Create missing Kafka topics.

    Parameters
    ----------
    topic_names : list[str]
        List of Kafka topic names
    broker_addr : str
        Kafka broker address.
    num_partitions : int
        Number of partitions for each topic.
    replication_factor : int
        Replication factor for each topic.
    """
    # Dict of kafka topic name: topic_info
    new_topics_info = [
        NewTopic(
            topic_name,
            num_partitions=num_partitions,
            replication_factor=replication_factor,
        )
        for topic_name in topic_names
    ]
    broker_client = AdminClient({"bootstrap.servers": broker_addr})
    if new_topics_info:
        create_result = broker_client.create_topics(new_topics_info)
        for topic_name, future in create_result.items():
            exception = future.exception()
            if exception is None:
                print(f"created topic {topic_name!r}")
                continue
            elif (
                isinstance(exception.args[0], KafkaError)
                and exception.args[0].code() == KafkaError.TOPIC_ALREADY_EXISTS
            ):
                print(f"topic {topic_name!r} already exists")
            else:
                print(f"Failed to create topic {topic_name!r}: {exception!r}")
                raise exception


def main():
    parser = argparse.ArgumentParser("Write Kafka messages")
    parser.add_argument(
        "-n", "--number", type=int, default=1, help="Number of messages to produce_data"
    )
    parser.add_argument(
        "--partitions",
        type=int,
        default=1,
        help="The number of partitions per topic.",
    )
    args = parser.parse_args()

    topic_name = "trivial.test.topic"

    acks = 0

    create_topics(
        topic_names=[topic_name],
        broker_addr=KAFKA_BROKER_ADDR,
        num_partitions=args.partitions,
    )

    print(f"Create a producer with {acks=}")
    producer = Producer(
        {
            "acks": acks,
            "queue.buffering.max.ms": 0,
            "bootstrap.servers": KAFKA_BROKER_ADDR,
            "api.version.request": True,
        }
    )
    producer.list_topics()
    print("before prod.")
    producer.flush()
    raw_data = b"some data"
    print(f"Produce 1 message")
    producer.produce(topic_name, raw_data)
    print("flush the first time.")
    t0 = time.time()
    producer.flush()
    t1 = time.time()
    dtflush = t1 - t0
    print(f"First flush took {dtflush:0.2f} seconds")
    return
    print(f"Produce {args.number} messages")
    for i in range(args.number):
        producer.produce(topic_name, raw_data)
        producer.flush()
    dt = time.time() - t1
    print(f"Wrote {args.number/dt:0.1f} messages/second: {args} (after the first message)")


main()
